/***********************************************************************\
*
* Contents: Backup ARchiver archive convert functions
* Systems: all
*
\***********************************************************************/

/****************************** Includes *******************************/
#include <config.h>  // use <...> to support separated build directory

#include <stdlib.h>
#include <stdio.h>
#include <pthread.h>
#include <semaphore.h>
#include <dirent.h>
#include <unistd.h>
#include <inttypes.h>
#include <errno.h>
#include <assert.h>

#include "common/global.h"
#include "common/cstrings.h"
#include "common/autofree.h"
#include "common/strings.h"
#include "common/stringlists.h"
#include "common/msgqueues.h"
#include "common/patterns.h"
#include "common/files.h"
#include "common/filesystems.h"

#include "bar.h"
#include "errors.h"
#include "archives.h"

#include "commands_convert.h"

/****************** Conditional compilation switches *******************/

/***************************** Constants *******************************/

// data buffer size
#define BUFFER_SIZE (64*1024)

// max. number of entry messages
#define MAX_ENTRY_MSG_QUEUE 256

/***************************** Datatypes *******************************/

// convert info
typedef struct
{
  StorageInfo             sourceStorageInfo;
  StorageInfo             destinationStorageInfo;
  const char              *newJobUUID;
  const char              *newEntityUUID;
  uint64                  newCreatedDateTime;
  const JobOptions        *newJobOptions;
  GetNamePasswordFunction getNamePasswordFunction;
  void                    *getNamePasswordUserData;
  LogHandle               *logHandle;                         // log handle

  String                  archiveName;                        // archive name (converted archive)
  String                  newArchiveName;                     // temporary new archive name (converted archive)
  ArchiveHandle           destinationArchiveHandle;

  MsgQueue                entryMsgQueue;                      // queue with entries to convert
  MsgQueue                storageMsgQueue;                    // queue with waiting storage files
  bool                    storageThreadExitFlag;

  Errors                  failError;                          // failure error
} ConvertInfo;

// entry message send to convert threads
typedef struct
{
  uint                archiveIndex;                        // still not used
  const ArchiveHandle *archiveHandle;
  ArchiveEntryTypes   archiveEntryType;
  ArchiveCryptInfo    *archiveCryptInfo;
  uint64              offset;                              // offset in archive
} EntryMsg;

// storage message, send from convert threads -> storage thread
typedef struct
{
  String intermediateFileName;                                // intermediate archive file name
  uint64 intermediateFileSize;                                // intermediate archive size [bytes]
} StorageMsg;

/***************************** Variables *******************************/

/****************************** Macros *********************************/

/***************************** Forwards ********************************/

/***************************** Functions *******************************/

#ifdef __cplusplus
  extern "C" {
#endif

/***********************************************************************\
* Name   : freeEntryMsg
* Purpose: free file entry message call back
* Input  : entryMsg - entry message
*          userData - user data (not used)
* Output : -
* Return : -
* Notes  : -
\***********************************************************************/

LOCAL void freeEntryMsg(EntryMsg *entryMsg, void *userData)
{
  assert(entryMsg != NULL);

  UNUSED_VARIABLE(entryMsg);
  UNUSED_VARIABLE(userData);
}

/***********************************************************************\
* Name   : freeStorageMsg
* Purpose: free storage msg
* Input  : storageMsg - storage message
*          userData   - user data (ignored)
* Output : -
* Return : -
* Notes  : -
\***********************************************************************/

LOCAL void freeStorageMsg(StorageMsg *storageMsg, void *userData)
{
  assert(storageMsg != NULL);

  UNUSED_VARIABLE(userData);

  String_delete(storageMsg->intermediateFileName);
}

/***********************************************************************\
* Name   : initConvertInfo
* Purpose: initialize convert info
* Input  : convertInfo             - convert info variable
*          newJobUUID              - new job UUID or NULL
*          newEntityUUID           - new entity UUID or NULL
*          newCreatedDateTime      - new created date/time or 0
*          newJobOptions           - new job options
*          getNamePasswordFunction - get password call back
*          getNamePasswordUserData - user data for get password call back
*          logHandle               - log handle (can be NULL)
* Output : convertInfo - initialized convert info variable
* Return : -
* Notes  : -
\***********************************************************************/

LOCAL void initConvertInfo(ConvertInfo             *convertInfo,
                           const char              *newJobUUID,
                           const char              *newEntityUUID,
                           uint64                  newCreatedDateTime,
                           const JobOptions        *newJobOptions,
                           GetNamePasswordFunction getNamePasswordFunction,
                           void                    *getNamePasswordUserData,
                           LogHandle               *logHandle
                          )
{
  assert(convertInfo != NULL);
  assert(newJobOptions != NULL);

  // init variables
  convertInfo->newJobUUID              = newJobUUID;
  convertInfo->newEntityUUID           = newEntityUUID;
  convertInfo->newCreatedDateTime      = newCreatedDateTime;
  convertInfo->newJobOptions           = newJobOptions;
  convertInfo->getNamePasswordFunction = getNamePasswordFunction;
  convertInfo->getNamePasswordUserData = getNamePasswordUserData;
  convertInfo->logHandle               = logHandle;
  convertInfo->archiveName             = String_new();
  convertInfo->newArchiveName          = String_new();
  convertInfo->failError               = ERROR_NONE;
  convertInfo->storageThreadExitFlag   = FALSE;

  // init entry name queue, storage queue
  if (!MsgQueue_init(&convertInfo->entryMsgQueue,
                     MAX_ENTRY_MSG_QUEUE,
                     CALLBACK_((MsgQueueMsgFreeFunction)freeEntryMsg,NULL)
                    )
     )
  {
    HALT_FATAL_ERROR("Cannot initialize entry message queue!");
  }
  if (!MsgQueue_init(&convertInfo->storageMsgQueue,
// TODO: 0?
                     0,
                     CALLBACK_((MsgQueueMsgFreeFunction)freeStorageMsg,NULL)
                    )
     )
  {
    HALT_FATAL_ERROR("Cannot initialize storage message queue!");
  }

#if 0
  // init locks
  if (!Semaphore_init(&convertInfo->storageInfoLock))
  {
    HALT_FATAL_ERROR("Cannot initialize storage semaphore!");
  }
#endif

  DEBUG_ADD_RESOURCE_TRACE(convertInfo,ConvertInfo);
}

/***********************************************************************\
* Name   : doneConvertInfo
* Purpose: deinitialize convert info
* Input  : convertInfo - convert info
* Output : -
* Return : -
* Notes  : -
\***********************************************************************/

LOCAL void doneConvertInfo(ConvertInfo *convertInfo)
{
  assert(convertInfo != NULL);

  DEBUG_REMOVE_RESOURCE_TRACE(convertInfo,ConvertInfo);

  MsgQueue_done(&convertInfo->storageMsgQueue);
  MsgQueue_done(&convertInfo->entryMsgQueue);
  String_delete(convertInfo->newArchiveName);
  String_delete(convertInfo->archiveName);
}

/***********************************************************************\
* Name   : archiveStore
* Purpose: call back to store archive file
* Input  : storageInfo          - storage info
*          jobUUID              - job UUID id
*          entityUUID           - entity UUID id
*          entityId             - index entity id
*          storageId            - index storage id
*          partNumber           - part number or ARCHIVE_PART_NUMBER_NONE
*                                 for single part
*          intermediateFileName - intermediate archive file name
*          intermediateFileSize - intermediate archive size [bytes]
*          userData             - user data
* Output : -
* Return : ERROR_NONE or error code
* Notes  : -
\***********************************************************************/

LOCAL Errors archiveStore(StorageInfo  *storageInfo,
                          IndexId      uuidId,
                          ConstString  jobUUID,
                          ConstString  entityUUID,
                          IndexId      entityId,
                          IndexId      storageId,
                          int          partNumber,
                          ConstString  intermediateFileName,
                          uint64       intermediateFileSize,
                          void         *userData
                         )
{
  Errors error;

  assert(storageInfo != NULL);
  assert(!String_isEmpty(intermediateFileName));

  UNUSED_VARIABLE(storageInfo);
  UNUSED_VARIABLE(uuidId);
  UNUSED_VARIABLE(jobUUID);
  UNUSED_VARIABLE(entityUUID);
  UNUSED_VARIABLE(entityId);
  UNUSED_VARIABLE(storageId);
  UNUSED_VARIABLE(partNumber);

  // get file info
// TODO replace by getFileSize()
  FileInfo fileInfo;
  error = File_getInfo(&fileInfo,intermediateFileName);
  if (error != ERROR_NONE)
  {
    return error;
  }

  // send to storage thread
  StorageMsg storageMsg;
  storageMsg.intermediateFileName = String_duplicate(intermediateFileName);
  storageMsg.intermediateFileSize = intermediateFileSize;
  DEBUG_TESTCODE() { freeStorageMsg(&storageMsg,NULL); return DEBUG_TESTCODE_ERROR(); }
  ConvertInfo *convertInfo = (ConvertInfo*)userData;
  assert(convertInfo != NULL);
  if (!MsgQueue_put(&convertInfo->storageMsgQueue,&storageMsg,sizeof(storageMsg)))
  {
    freeStorageMsg(&storageMsg,NULL);
    (void)File_delete(intermediateFileName,FALSE);
    return ERROR_ABORTED;
  }

//TODO
#if 0
  // wait for space in temporary directory
  if (globalOptions.maxTmpSize > 0)
  {
    SEMAPHORE_LOCKED_DO(semaphoreLock,&convertInfo->storageInfoLock,SEMAPHORE_LOCK_TYPE_READ,WAIT_FOREVER)
    {
      while (   (convertInfo->storage.count > 2)                           // more than 2 archives are waiting
             && (convertInfo->storage.bytes > globalOptions.maxTmpSize)    // temporary space limit exceeded
             && (convertInfo->failError == ERROR_NONE)
             && !isAborted(convertInfo)
            )
      {
        Semaphore_waitModified(&convertInfo->storageInfoLock,30*1000);
      }
    }
  }
#endif

  // free resources

  return ERROR_NONE;
}

/***********************************************************************\
* Name   : storageThreadCode
* Purpose: archive storage thread
* Input  : convertInfo - convert info block
* Output : -
* Return : -
* Notes  : -
\***********************************************************************/

LOCAL void storageThreadCode(ConvertInfo *convertInfo)
{
  #define MAX_RETRIES 3

  assert(convertInfo != NULL);
  assert(convertInfo->newJobOptions != NULL);

  // init variables
  AutoFreeList autoFreeList;
  AutoFree_init(&autoFreeList);

  // store archives
  byte   *buffer = (byte*)malloc(BUFFER_SIZE);
  if (buffer == NULL)
  {
    HALT_INSUFFICIENT_MEMORY();
  }
  AUTOFREE_ADD(&autoFreeList,buffer,{ free(buffer); });
  String printableStorageName = String_new();
  AUTOFREE_ADD(&autoFreeList,printableStorageName,{ String_delete(printableStorageName); });
  String tmpArchiveName       = String_new();
  AUTOFREE_ADD(&autoFreeList,tmpArchiveName,{ String_delete(tmpArchiveName); });
  while (   (convertInfo->failError == ERROR_NONE)
//         && !isAborted(convertInfo)
        )
  {
    void *autoFreeSavePoint = AutoFree_save(&autoFreeList);

    // get next archive to store
    StorageMsg storageMsg;
    if (!MsgQueue_get(&convertInfo->storageMsgQueue,&storageMsg,NULL,sizeof(storageMsg),WAIT_FOREVER))
    {
      break;
    }
    AUTOFREE_ADD(&autoFreeList,&storageMsg,
                 {
                   File_delete(storageMsg.intermediateFileName,FALSE);
                   freeStorageMsg(&storageMsg,NULL);
                 }
                );

    Errors error;

    // get file info
    FileInfo fileInfo;
    error = File_getInfo(&fileInfo,storageMsg.intermediateFileName);
    if (error != ERROR_NONE)
    {
      printError(_("cannot get information for file '%s' (error: %s)!"),
                 String_cString(storageMsg.intermediateFileName),
                 Error_getText(error)
                );
      convertInfo->failError = error;

      AutoFree_restore(&autoFreeList,autoFreeSavePoint,TRUE);
      continue;
    }
    DEBUG_TESTCODE() { convertInfo->failError = DEBUG_TESTCODE_ERROR(); AutoFree_restore(&autoFreeList,autoFreeSavePoint,TRUE); break; }

    // get printable storage name
    Storage_getPrintableName(printableStorageName,&convertInfo->sourceStorageInfo.storageSpecifier,convertInfo->archiveName);

    // open file to store
    #ifndef NDEBUG
      printInfo(1,"Store '%s' to '%s'...",String_cString(storageMsg.intermediateFileName),String_cString(printableStorageName));
    #else /* not NDEBUG */
      printInfo(1,"Store '%s'...",String_cString(printableStorageName));
    #endif /* NDEBUG */
    FileHandle fileHandle;
    error = File_open(&fileHandle,storageMsg.intermediateFileName,FILE_OPEN_READ);
    if (error != ERROR_NONE)
    {
      printInfo(0,"FAIL!\n");
      printError(_("cannot open file '%s' (error: %s)!"),
                 String_cString(storageMsg.intermediateFileName),
                 Error_getText(error)
                );
      convertInfo->failError = error;
      AutoFree_restore(&autoFreeList,autoFreeSavePoint,TRUE);
      continue;
    }
    AUTOFREE_ADD(&autoFreeList,&fileHandle,{ File_close(&fileHandle); });
    DEBUG_TESTCODE() { convertInfo->failError = DEBUG_TESTCODE_ERROR(); AutoFree_restore(&autoFreeList,autoFreeSavePoint,TRUE); continue; }

    // rename original storage file to temporary file
    if (Storage_exists(&convertInfo->sourceStorageInfo,convertInfo->archiveName))
    {
      Storage_getTmpName(tmpArchiveName,&convertInfo->sourceStorageInfo);
//fprintf(stderr,"%s, %d: rename original %s -> %s\n",__FILE__,__LINE__,String_cString(convertInfo->archiveName),String_cString(tmpArchiveName));
      error = Storage_rename(&convertInfo->sourceStorageInfo,convertInfo->archiveName,tmpArchiveName);
      if (error != ERROR_NONE)
      {
        printInfo(0,"FAIL!\n");
        printError(_("cannot store '%s' (error: %s)"),
                   String_cString(printableStorageName),
                   Error_getText(error)
                  );
        convertInfo->failError = error;
        AutoFree_restore(&autoFreeList,autoFreeSavePoint,TRUE);
        continue;
      }
      AUTOFREE_ADD(&autoFreeList,&tmpArchiveName, { Storage_rename(&convertInfo->sourceStorageInfo,tmpArchiveName,convertInfo->archiveName); });
    }
    else
    {
      String_clear(tmpArchiveName);
    }

    // create storage file
    if (!String_isEmpty(convertInfo->newJobOptions->destination))
    {
// TODO: create correct destination name

//TODO: remove?
      assert(convertInfo->destinationArchiveHandle.mode == ARCHIVE_MODE_CREATE);

      // create local file
      FileHandle toFileHandle;
      error = File_open(&toFileHandle,convertInfo->archiveName,FILE_OPEN_CREATE);
      if (error != ERROR_NONE)
      {
        printInfo(0,"FAIL!\n");
        printError(_("cannot store '%s' (error: %s)"),
                   String_cString(convertInfo->destinationArchiveHandle.archiveName),
                   Error_getText(error)
                  );
        break;
      }
      DEBUG_TESTCODE() { File_close(&toFileHandle); error = DEBUG_TESTCODE_ERROR(); break; }
      AUTOFREE_ADD(&autoFreeList,&toFileHandle,{ File_close(&toFileHandle); });

      // transfer data from temporary file to local file
      File_seek(&fileHandle,0);
      do
      {
        // read data from local intermediate file
        ulong bufferLength;
        error = File_read(&fileHandle,buffer,BUFFER_SIZE,&bufferLength);
        if (error != ERROR_NONE)
        {
          printInfo(0,"FAIL!\n");
          printError(_("cannot read file '%s' (error: %s)!"),
                     String_cString(printableStorageName),
                     Error_getText(error)
                    );
          break;
        }
        DEBUG_TESTCODE() { error = DEBUG_TESTCODE_ERROR(); break; }

        // store data into local file
        error = File_write(&toFileHandle,buffer,bufferLength);
        if (error != ERROR_NONE)
        {
          printInfo(0,"FAIL!\n");
          printError(_("cannot write file '%s' (error: %s)!"),
                     String_cString(printableStorageName),
                     Error_getText(error)
                    );
          break;
        }
        DEBUG_TESTCODE() { error = DEBUG_TESTCODE_ERROR(); break; }
      }
      while (   (convertInfo->failError == ERROR_NONE)
//             && !isAborted(convertInfo)
             && !File_eof(&fileHandle)
            );

      // close local file
      File_close(&toFileHandle);
      AUTOFREE_REMOVE(&autoFreeList,&toFileHandle);
    }
    else
    {
      uint retryCount = 0;
      do
      {
        // next try
        if (retryCount > MAX_RETRIES)
        {
          break;
        }
        retryCount++;

        // create storage file
        StorageHandle storageHandle;
        error = Storage_create(&storageHandle,
                               &convertInfo->destinationStorageInfo,
                               convertInfo->archiveName,
                               fileInfo.size,
                               TRUE  // forceFlag
                              );
        if (error != ERROR_NONE)
        {
          if (retryCount <= MAX_RETRIES)
          {
            // retry
            continue;
          }
          else
          {
            printInfo(0,"FAIL!\n");
            printError(_("cannot store '%s' (error: %s)"),
                       String_cString(printableStorageName),
                       Error_getText(error)
                      );
            break;
          }
        }
        DEBUG_TESTCODE() { Storage_close(&storageHandle); error = DEBUG_TESTCODE_ERROR(); break; }
        AUTOFREE_ADD(&autoFreeList,&storageHandle,{ Storage_close(&storageHandle); Storage_delete(&convertInfo->destinationStorageInfo,convertInfo->archiveName); });

        // transfer data from temporary file to storage
        File_seek(&fileHandle,0);
        do
        {
          // read data from local intermediate file
          ulong bufferLength;
          error = File_read(&fileHandle,buffer,BUFFER_SIZE,&bufferLength);
          if (error != ERROR_NONE)
          {
            printInfo(0,"FAIL!\n");
            printError(_("cannot read file '%s' (error: %s)!"),
                       String_cString(printableStorageName),
                       Error_getText(error)
                      );
            break;
          }
          DEBUG_TESTCODE() { error = DEBUG_TESTCODE_ERROR(); break; }

          // store data into storage
          error = Storage_write(&storageHandle,buffer,bufferLength);
          if (error != ERROR_NONE)
          {
            if (retryCount <= MAX_RETRIES)
            {
              // retry
              break;
            }
            else
            {
              printInfo(0,"FAIL!\n");
              printError(_("cannot write file '%s' (error: %s)!"),
                         String_cString(printableStorageName),
                         Error_getText(error)
                        );
              break;
            }
          }
          DEBUG_TESTCODE() { error = DEBUG_TESTCODE_ERROR(); break; }
        }
        while (   (convertInfo->failError == ERROR_NONE)
  //             && !isAborted(convertInfo)
               && !File_eof(&fileHandle)
              );

        // close local file
        Storage_close(&storageHandle);
        AUTOFREE_REMOVE(&autoFreeList,&storageHandle);

        DEBUG_TESTCODE() { AutoFree_cleanup(&autoFreeList); error = DEBUG_TESTCODE_ERROR(); }
      }
      while (   (convertInfo->failError == ERROR_NONE)                           // no eror
  //           && !isAborted(convertInfo)                                        // not aborted
             && ((error != ERROR_NONE) && (Error_getErrno(error) != ENOSPC))     // some error and not "no space left"
             && (retryCount <= MAX_RETRIES)                                      // still some retry left
            );
    }
    if (error != ERROR_NONE)
    {
      if (convertInfo->failError == ERROR_NONE) convertInfo->failError = error;
      AutoFree_restore(&autoFreeList,autoFreeSavePoint,TRUE);
      continue;
    }

    // close file
    File_close(&fileHandle);
    AUTOFREE_REMOVE(&autoFreeList,&fileHandle);

#if 0
    // check if aborted
    if (isAborted(convertInfo))
    {
      printInfo(1,"ABORTED\n");
      AutoFree_restore(&autoFreeList,autoFreeSavePoint,TRUE);
      continue;
    }
#endif

    if (!String_isEmpty(tmpArchiveName))
    {
      // delete original storage file
//fprintf(stderr,"%s, %d: delete saved orginal %s\n",__FILE__,__LINE__,String_cString(tmpArchiveName));
      error = Storage_delete(&convertInfo->sourceStorageInfo,tmpArchiveName);
      if (error != ERROR_NONE)
      {
        printInfo(0,"FAIL!\n");
        printError(_("cannot store '%s' (error: %s)"),
                   String_cString(printableStorageName),
                   Error_getText(error)
                  );
        convertInfo->failError = error;
        AutoFree_restore(&autoFreeList,autoFreeSavePoint,TRUE);
        continue;
      }
      AUTOFREE_REMOVE(&autoFreeList,&tmpArchiveName);
    }

    // done
    printInfo(1,"OK\n");

    // delete temporary storage file
    error = File_delete(storageMsg.intermediateFileName,FALSE);
    if (error != ERROR_NONE)
    {
      printWarning(_("cannot delete file '%s' (error: %s)!"),
                   String_cString(storageMsg.intermediateFileName),
                   Error_getText(error)
                  );
    }

    // free resources
    freeStorageMsg(&storageMsg,NULL);

    AutoFree_restore(&autoFreeList,autoFreeSavePoint,FALSE);
  }
  String_delete(tmpArchiveName);
  String_delete(printableStorageName);
  free(buffer);

  // discard unprocessed archives
  StorageMsg storageMsg;
  while (MsgQueue_get(&convertInfo->storageMsgQueue,&storageMsg,NULL,sizeof(storageMsg),NO_WAIT))
  {
    // delete temporary storage file
    Errors error = File_delete(storageMsg.intermediateFileName,FALSE);
    if (error != ERROR_NONE)
    {
      printWarning(_("cannot delete file '%s' (error: %s)!"),
                   String_cString(storageMsg.intermediateFileName),
                   Error_getText(error)
                  );
    }

    // free resources
    freeStorageMsg(&storageMsg,NULL);
  }

  // free resoures
  AutoFree_done(&autoFreeList);

  convertInfo->storageThreadExitFlag = TRUE;
}

/***********************************************************************\
* Name   : convertFileEntry
* Purpose: convert a file entry in archive
* Input  : sourceArchiveHandle      - source archive handle
*          destinationArchiveHandle - destination archive handle
*          newJobOptions            - new job options
*          buffer                   - buffer for temporary data
*          bufferSize               - size of data buffer
* Output : -
* Return : ERROR_NONE or error code
* Notes  : -
\***********************************************************************/

LOCAL Errors convertFileEntry(ArchiveHandle    *sourceArchiveHandle,
                              ArchiveHandle    *destinationArchiveHandle,
                              const JobOptions *newJobOptions,
                              byte             *buffer,
                              uint             bufferSize
                             )
{
  Errors error;

  // read source file entry
  ArchiveEntryInfo          sourceArchiveEntryInfo;
  CompressAlgorithms        deltaCompressAlgorithm,byteCompressAlgorithm;
  CryptAlgorithms           cryptAlgorithm;
  const CryptSalt           *cryptSalt;
  const CryptKey            *cryptKey;
  String                    fileName = String_new();
  FileInfo                  fileInfo;
  FileExtendedAttributeList fileExtendedAttributeList;
  File_initExtendedAttributes(&fileExtendedAttributeList);
  uint64                    fragmentOffset,fragmentSize;
  error = Archive_readFileEntry(&sourceArchiveEntryInfo,
                                sourceArchiveHandle,
                                &deltaCompressAlgorithm,
                                &byteCompressAlgorithm,
                                NULL,  // cryptType,
                                &cryptAlgorithm,
                                &cryptSalt,
                                &cryptKey,
                                fileName,
                                &fileInfo,
                                &fileExtendedAttributeList,
                                NULL,  // deltaSourceName
                                NULL,  // deltaSourceSize
                                &fragmentOffset,
                                &fragmentSize
                               );
  if (error != ERROR_NONE)
  {
    printError(_("cannot read 'file' entry from storage '%s' (error: %s)!"),
               String_cString(sourceArchiveHandle->printableStorageName),
               Error_getText(error)
              );
    File_doneExtendedAttributes(&fileExtendedAttributeList);
    String_delete(fileName);
    return error;
  }
  DEBUG_TESTCODE() { Archive_closeEntry(&sourceArchiveEntryInfo); File_doneExtendedAttributes(&fileExtendedAttributeList); String_delete(fileName); return DEBUG_TESTCODE_ERROR(); }

  // get size/fragment info
  char sizeString[32];
  if (globalOptions.humanFormatFlag)
  {
    getHumanSizeString(sizeString,sizeof(sizeString),fileInfo.size);
  }
  else
  {
    stringFormat(sizeString,sizeof(sizeString),"%"PRIu64,fileInfo.size);
  }
  char fragmentString[256];
  stringClear(fragmentString);
  if (fragmentSize < fileInfo.size)
  {
    stringFormat(fragmentString,sizeof(fragmentString),
                 ", fragment %*"PRIu64"..%*"PRIu64,
                 stringInt64Length(fileInfo.size),fragmentOffset,
                 stringInt64Length(fileInfo.size),fragmentOffset+fragmentSize-1LL
                );
  }
  printInfo(1,"  Convert file      '%s' (%s bytes%s)...",String_cString(fileName),sizeString,fragmentString);

  // set new compression, crypt settings
  if (Configuration_isCommandLineOptionSet(&globalOptions.compressAlgorithms)) byteCompressAlgorithm = newJobOptions->compressAlgorithms.byte;
  if (Configuration_isCommandLineOptionSet(globalOptions.cryptAlgorithms    )) cryptAlgorithm = newJobOptions->cryptAlgorithms[0];
  if (Configuration_isCommandLineOptionSet(globalOptions.cryptAlgorithms) || Configuration_isCommandLineOptionSet(&globalOptions.cryptNewPassword))
  {
    cryptSalt = NULL;
    cryptKey  = NULL;
  }

  // check if file data should be byte compressed
  ArchiveFlags archiveFlags = ARCHIVE_FLAG_NONE;
  if (   (fileInfo.size > globalOptions.compressMinFileSize)
//TODO
//      && !PatternList_match(newJobOptions->compressExcludePatternList,fileName,PATTERN_MATCH_MODE_EXACT)
     )
  {
     archiveFlags |= ARCHIVE_FLAG_TRY_BYTE_COMPRESS;
  }

  // create new file entry
  ArchiveEntryInfo destinationArchiveEntryInfo;
  error = Archive_newFileEntry(&destinationArchiveEntryInfo,
                               destinationArchiveHandle,
                               deltaCompressAlgorithm,
                               byteCompressAlgorithm,
                               cryptAlgorithm,
                               cryptSalt,
                               cryptKey,
                               fileName,
                               &fileInfo,
                               &fileExtendedAttributeList,
                               fragmentOffset,
                               fragmentSize,
                               archiveFlags
                              );
  if (error != ERROR_NONE)
  {
    printError(_("cannot create new archive file '%s' (error: %s)"),
               String_cString(fileName),
               Error_getText(error)
              );
    (void)Archive_closeEntry(&sourceArchiveEntryInfo);
    File_doneExtendedAttributes(&fileExtendedAttributeList);
    String_delete(fileName);
    return error;
  }
  DEBUG_TESTCODE() { Archive_closeEntry(&destinationArchiveEntryInfo); Archive_closeEntry(&sourceArchiveEntryInfo); File_doneExtendedAttributes(&fileExtendedAttributeList); String_delete(fileName); return DEBUG_TESTCODE_ERROR(); }

  // convert archive and file content
  uint64 length = 0LL;
  while (length < fragmentSize)
  {
    ulong bufferLength = (ulong)MIN(fragmentSize-length,bufferSize);

    // read source archive
    error = Archive_readData(&sourceArchiveEntryInfo,buffer,bufferLength);
    if (error != ERROR_NONE)
    {
      printInfo(1,"FAIL!\n");
      printError(_("cannot read content of 'file' entry '%s' (error: %s)!"),
                 String_cString(fileName),
                 Error_getText(error)
                );
      break;
    }
    DEBUG_TESTCODE() { error = DEBUG_TESTCODE_ERROR(); break; }

    // write data to destination archive
    error = Archive_writeData(&destinationArchiveEntryInfo,buffer,bufferLength,1);
    if (error != ERROR_NONE)
    {
      printInfo(0,"FAIL!\n");
      printError(_("cannot write content of 'file' entry '%s' (error: %s)!"),
                 String_cString(fileName),
                 Error_getText(error)
                );
      break;
    }
    DEBUG_TESTCODE() { error = DEBUG_TESTCODE_ERROR(); break; }

    length += (uint64)bufferLength;

    printInfo(2,"%3d%%\b\b\b\b",(uint)((length*100LL)/fragmentSize));
  }
  if (error != ERROR_NONE)
  {
    (void)Archive_closeEntry(&destinationArchiveEntryInfo);
    (void)Archive_closeEntry(&sourceArchiveEntryInfo);
    File_doneExtendedAttributes(&fileExtendedAttributeList);
    String_delete(fileName);
    return error;
  }
  DEBUG_TESTCODE() { (void)Archive_closeEntry(&destinationArchiveEntryInfo); (void)Archive_closeEntry(&sourceArchiveEntryInfo); File_doneExtendedAttributes(&fileExtendedAttributeList); String_delete(fileName); return DEBUG_TESTCODE_ERROR(); }

  printInfo(2,"    \b\b\b\b");

  // close destination archive entry
  error = Archive_closeEntry(&destinationArchiveEntryInfo);
  if (error != ERROR_NONE)
  {
    printInfo(1,"FAIL\n");
    printError(_("cannot close archive file entry '%s' (error: %s)!"),
               String_cString(fileName),
               Error_getText(error)
              );
    (void)Archive_closeEntry(&sourceArchiveEntryInfo);
    File_doneExtendedAttributes(&fileExtendedAttributeList);
    String_delete(fileName);
    return error;
  }

  printInfo(1,"OK\n");

  /* check if all data read.
     Note: it is not possible to check if all data is read when
     compression is used. The decompressor may not be at the end
     of a compressed data chunk even compressed data is _not_
     corrupt.
  */
  if (   !Compress_isCompressed(deltaCompressAlgorithm)
      && !Compress_isCompressed(byteCompressAlgorithm)
      && !Archive_eofData(&sourceArchiveEntryInfo))
  {
    printWarning(_("unexpected data at end of file entry '%s'"),String_cString(fileName));
  }

  // close source archive entry
  error = Archive_closeEntry(&sourceArchiveEntryInfo);
  if (error != ERROR_NONE)
  {
    printWarning(_("close 'file' entry fail (error: %s)"),Error_getText(error));
  }

  // free resources
  File_doneExtendedAttributes(&fileExtendedAttributeList);
  String_delete(fileName);

  return ERROR_NONE;
}

/***********************************************************************\
* Name   : convertImageEntry
* Purpose: convert a image entry in archive
* Input  : sourceArchiveHandle      - source archive handle
*          destinationArchiveHandle - destination archive handle
*          newJobOptions            - new job options
*          buffer                   - buffer for temporary data
*          bufferSize               - size of data buffer
* Output : -
* Return : ERROR_NONE or error code
* Notes  : -
\***********************************************************************/

LOCAL Errors convertImageEntry(ArchiveHandle    *sourceArchiveHandle,
                               ArchiveHandle    *destinationArchiveHandle,
                               const JobOptions *newJobOptions,
                               byte             *buffer,
                               uint             bufferSize
                              )
{
  Errors error;

  // read source image entry
  ArchiveEntryInfo   sourceArchiveEntryInfo;
  CompressAlgorithms deltaCompressAlgorithm,byteCompressAlgorithm;
  CryptAlgorithms    cryptAlgorithm;
  const CryptSalt    *cryptSalt;
  const CryptKey     *cryptKey;
  String             deviceName = String_new();
  DeviceInfo         deviceInfo;
  FileSystemTypes    fileSystemType;
  uint64             blockOffset,blockCount;
  error = Archive_readImageEntry(&sourceArchiveEntryInfo,
                                 sourceArchiveHandle,
                                 &deltaCompressAlgorithm,
                                 &byteCompressAlgorithm,
                                 NULL,  // cryptType,
                                 &cryptAlgorithm,
                                 &cryptSalt,
                                 &cryptKey,
                                 deviceName,
                                 &deviceInfo,
                                 &fileSystemType,
                                 NULL,  // deltaSourceName
                                 NULL,  // deltaSourceSize
                                 &blockOffset,
                                 &blockCount
                                );
  if (error != ERROR_NONE)
  {
    printError(_("cannot read 'image' entry from storage '%s' (error: %s)!"),
               String_cString(sourceArchiveHandle->printableStorageName),
               Error_getText(error)
              );
    String_delete(deviceName);
    return error;
  }
  DEBUG_TESTCODE() { Archive_closeEntry(&sourceArchiveEntryInfo); String_delete(deviceName); return DEBUG_TESTCODE_ERROR(); }
  if (deviceInfo.blockSize > bufferSize)
  {
    printError(_("device block size %"PRIu64" on '%s' is too big (max: %"PRIu64")"),
               deviceInfo.blockSize,
               String_cString(deviceName),
               bufferSize
              );
    String_delete(deviceName);
    return error;
  }
  DEBUG_TESTCODE() { Archive_closeEntry(&sourceArchiveEntryInfo); String_delete(deviceName); return DEBUG_TESTCODE_ERROR(); }
  assert(deviceInfo.blockSize > 0);

  // get size/fragment info
  char sizeString[32];
  if (globalOptions.humanFormatFlag)
  {
    getHumanSizeString(sizeString,sizeof(sizeString),deviceInfo.size);
  }
  else
  {
    stringFormat(sizeString,sizeof(sizeString),"%"PRIu64,blockCount*(uint64)deviceInfo.blockSize);
  }
  char fragmentString[256];
  stringClear(fragmentString);
  if ((blockCount*(uint64)deviceInfo.blockSize) < deviceInfo.size)
  {
    stringFormat(fragmentString,sizeof(fragmentString),
                 ", fragment %*"PRIu64"..%*"PRIu64,
                 stringInt64Length(deviceInfo.size),blockOffset*deviceInfo.blockSize,
                 stringInt64Length(deviceInfo.size),blockOffset*deviceInfo.blockSize+(blockCount*deviceInfo.blockSize)-1LL
                );
  }
  printInfo(1,"  Convert image     '%s' (%s bytes%s)...",String_cString(deviceName),sizeString,fragmentString);

  // set new compression, crypt settings
  if (Configuration_isCommandLineOptionSet(&globalOptions.compressAlgorithms)) byteCompressAlgorithm = newJobOptions->compressAlgorithms.byte;
  if (Configuration_isCommandLineOptionSet(globalOptions.cryptAlgorithms    )) cryptAlgorithm        = newJobOptions->cryptAlgorithms[0];
  if (Configuration_isCommandLineOptionSet(globalOptions.cryptAlgorithms) || Configuration_isCommandLineOptionSet(&globalOptions.cryptNewPassword))
  {
    cryptSalt = NULL;
    cryptKey  = NULL;
  }

  // check if file data should be byte compressed
  ArchiveFlags archiveFlags = ARCHIVE_FLAG_NONE;
  if (   (deviceInfo.size > globalOptions.compressMinFileSize)
//TODO
//      && !PatternList_match(createInfo->compressExcludePatternList,fileName,PATTERN_MATCH_MODE_EXACT)
     )
  {
     archiveFlags |= ARCHIVE_FLAG_TRY_BYTE_COMPRESS;
  }

  // create new image entry
  ArchiveEntryInfo destinationArchiveEntryInfo;
  error = Archive_newImageEntry(&destinationArchiveEntryInfo,
                                destinationArchiveHandle,
                                deltaCompressAlgorithm,
                                byteCompressAlgorithm,
                                cryptAlgorithm,
                                cryptSalt,
                                cryptKey,
                                deviceName,
                                &deviceInfo,
                                fileSystemType,
                                blockOffset,
                                blockCount,
                                archiveFlags
                               );
  if (error != ERROR_NONE)
  {
    printError(_("cannot create new archive image entry '%s' (error: %s)!"),
               String_cString(deviceName),
               Error_getText(error)
              );
    (void)Archive_closeEntry(&sourceArchiveEntryInfo);
    String_delete(deviceName);
    return error;
  }
  DEBUG_TESTCODE() { Archive_closeEntry(&destinationArchiveEntryInfo); Archive_closeEntry(&sourceArchiveEntryInfo); String_delete(deviceName); return DEBUG_TESTCODE_ERROR(); }

  // convert archive and device/image content
  uint64 block = 0LL;
  while (block < blockCount)
  {
    // read data from archive (only single block)
    error = Archive_readData(&sourceArchiveEntryInfo,buffer,deviceInfo.blockSize);
    if (error != ERROR_NONE)
    {
      printInfo(1,"FAIL!\n");
      printError(_("cannot read content of 'image' entry '%s' (error: %s)!"),
                 String_cString(deviceName),
                 Error_getText(error)
                );
      break;
    }
    DEBUG_TESTCODE() { error = DEBUG_TESTCODE_ERROR(); break; }

    // write data to destination archive
    error = Archive_writeData(&destinationArchiveEntryInfo,buffer,deviceInfo.blockSize,1);
    if (error != ERROR_NONE)
    {
      printInfo(0,"FAIL!\n");
      printError(_("cannot write content of 'image' entry '%s' (error: %s)!"),
                 String_cString(deviceName),
                 Error_getText(error)
                );
      break;
    }
    DEBUG_TESTCODE() { error = DEBUG_TESTCODE_ERROR(); break; }

    block += 1LL;

    printInfo(2,"%3d%%\b\b\b\b",(uint)((block*100LL)/blockCount));
  }
  if (error != ERROR_NONE)
  {
    (void)Archive_closeEntry(&destinationArchiveEntryInfo);
    (void)Archive_closeEntry(&sourceArchiveEntryInfo);
    String_delete(deviceName);
    return error;
  }
  DEBUG_TESTCODE() { (void)Archive_closeEntry(&destinationArchiveEntryInfo); (void)Archive_closeEntry(&sourceArchiveEntryInfo); String_delete(deviceName); return DEBUG_TESTCODE_ERROR(); }

  printInfo(2,"    \b\b\b\b");

  // close destination archive entry
  error = Archive_closeEntry(&destinationArchiveEntryInfo);
  if (error != ERROR_NONE)
  {
    printInfo(1,"FAIL\n");
    printError(_("cannot close archive image entry (error: %s)!"),
               Error_getText(error)
              );
    (void)Archive_closeEntry(&sourceArchiveEntryInfo);
    String_delete(deviceName);
    return error;
  }

  printInfo(1,"OK\n");

  /* check if all data read.
     Note: it is not possible to check if all data is read when
     compression is used. The decompressor may not be at the end
     of a compressed data chunk even compressed data is _not_
     corrupt.
  */
  if (   !Compress_isCompressed(deltaCompressAlgorithm)
      && !Compress_isCompressed(byteCompressAlgorithm)
      && !Archive_eofData(&sourceArchiveEntryInfo))
  {
    printWarning(_("unexpected data at end of image entry '%s'"),String_cString(deviceName));
  }

  // close source archive entry
  error = Archive_closeEntry(&sourceArchiveEntryInfo);
  if (error != ERROR_NONE)
  {
    printWarning(_("close 'image' entry fail (error: %s)"),Error_getText(error));
    // ignore error
  }

  // free resources
  String_delete(deviceName);

  return ERROR_NONE;
}

/***********************************************************************\
* Name   : convertDirectoryEntry
* Purpose: convert a directory entry in archive
* Input  : sourceArchiveHandle      - source archive handle
*          destinationArchiveHandle - destination archive handle
*          newJobOptions            - new job options
* Output : -
* Return : ERROR_NONE or error code
* Notes  : -
\***********************************************************************/

LOCAL Errors convertDirectoryEntry(ArchiveHandle    *sourceArchiveHandle,
                                   ArchiveHandle    *destinationArchiveHandle,
                                   const JobOptions *newJobOptions
                                  )
{
  Errors error;

  // read source directory entry
  ArchiveEntryInfo          sourceArchiveEntryInfo;
  CryptAlgorithms           cryptAlgorithm;
  const CryptSalt           *cryptSalt;
  const CryptKey            *cryptKey;
  String                    directoryName = String_new();
  FileInfo                  fileInfo;
  FileExtendedAttributeList fileExtendedAttributeList;
  File_initExtendedAttributes(&fileExtendedAttributeList);
  error = Archive_readDirectoryEntry(&sourceArchiveEntryInfo,
                                     sourceArchiveHandle,
                                     NULL,  // cryptType,
                                     &cryptAlgorithm,
                                     &cryptSalt,
                                     &cryptKey,
                                     directoryName,
                                     &fileInfo,
                                     &fileExtendedAttributeList
                                    );
  if (error != ERROR_NONE)
  {
    printError(_("cannot read 'directory' entry from storage '%s' (error: %s)!"),
               String_cString(sourceArchiveHandle->printableStorageName),
               Error_getText(error)
              );
    File_doneExtendedAttributes(&fileExtendedAttributeList);
    String_delete(directoryName);
    return error;
  }
  DEBUG_TESTCODE() { Archive_closeEntry(&sourceArchiveEntryInfo); File_doneExtendedAttributes(&fileExtendedAttributeList); String_delete(directoryName); return DEBUG_TESTCODE_ERROR(); }

  printInfo(1,"  Convert directory '%s'...",String_cString(directoryName));

  // set new crypt settings
  if (Configuration_isCommandLineOptionSet(globalOptions.cryptAlgorithms)) cryptAlgorithm = newJobOptions->cryptAlgorithms[0];
  if (Configuration_isCommandLineOptionSet(globalOptions.cryptAlgorithms) || Configuration_isCommandLineOptionSet(&globalOptions.cryptNewPassword))
  {
    cryptSalt = NULL;
    cryptKey  = NULL;
  }

  // create new directory entry
  ArchiveEntryInfo destinationArchiveEntryInfo;
  error = Archive_newDirectoryEntry(&destinationArchiveEntryInfo,
                                    destinationArchiveHandle,
                                    cryptAlgorithm,
                                    cryptSalt,
                                    cryptKey,
                                    directoryName,
                                    &fileInfo,
                                    &fileExtendedAttributeList
                                   );
  if (error != ERROR_NONE)
  {
    printError(_("cannot create new archive directory entry '%s' (error: %s)"),
               String_cString(directoryName),
               Error_getText(error)
              );
    (void)Archive_closeEntry(&sourceArchiveEntryInfo);
    File_doneExtendedAttributes(&fileExtendedAttributeList);
    String_delete(directoryName);
    return error;
  }
  DEBUG_TESTCODE() { Archive_closeEntry(&destinationArchiveEntryInfo); (void)Archive_closeEntry(&sourceArchiveEntryInfo); File_doneExtendedAttributes(&fileExtendedAttributeList); String_delete(directoryName); return DEBUG_TESTCODE_ERROR(); }

  // close destination archive entry
  error = Archive_closeEntry(&destinationArchiveEntryInfo);
  if (error != ERROR_NONE)
  {
    printInfo(1,"FAIL\n");
    printError(_("cannot close archive directory entry (error: %s)!"),
               Error_getText(error)
              );
    (void)Archive_closeEntry(&sourceArchiveEntryInfo);
    File_doneExtendedAttributes(&fileExtendedAttributeList);
    String_delete(directoryName);
    return error;
  }

  printInfo(1,"OK\n");

  // check if all data read
  if (!Archive_eofData(&sourceArchiveEntryInfo))
  {
    printWarning(_("unexpected data at end of directory entry '%s'"),String_cString(directoryName));
  }

  // close source archive entry
  error = Archive_closeEntry(&sourceArchiveEntryInfo);
  if (error != ERROR_NONE)
  {
    printWarning(_("close 'directory' entry fail (error: %s)"),Error_getText(error));
  }

  // free resources
  File_doneExtendedAttributes(&fileExtendedAttributeList);
  String_delete(directoryName);

  return ERROR_NONE;
}

/***********************************************************************\
* Name   : convertLinkEntry
* Purpose: convert a link entry in archive
* Input  : sourceArchiveHandle      - source archive handle
*          destinationArchiveHandle - destination archive handle
*          newJobOptions            - new job options
* Output : -
* Return : ERROR_NONE or error code
* Notes  : -
\***********************************************************************/

LOCAL Errors convertLinkEntry(ArchiveHandle    *sourceArchiveHandle,
                              ArchiveHandle    *destinationArchiveHandle,
                              const JobOptions *newJobOptions
                             )
{
  Errors error;

  // read source link entry
  ArchiveEntryInfo          sourceArchiveEntryInfo;
  String                    linkName = String_new();
  String                    fileName = String_new();
  CryptAlgorithms           cryptAlgorithm;
  const CryptSalt           *cryptSalt;
  const CryptKey            *cryptKey;
  FileExtendedAttributeList fileExtendedAttributeList;
  File_initExtendedAttributes(&fileExtendedAttributeList);
  FileInfo                  fileInfo;
  error = Archive_readLinkEntry(&sourceArchiveEntryInfo,
                                sourceArchiveHandle,
                                NULL,  // cryptType,
                                &cryptAlgorithm,
                                &cryptSalt,
                                &cryptKey,
                                linkName,
                                fileName,
                                &fileInfo,
                                &fileExtendedAttributeList
                               );
  if (error != ERROR_NONE)
  {
    printError(_("cannot read 'link' entry from storage '%s' (error: %s)!"),
               String_cString(sourceArchiveHandle->printableStorageName),
               Error_getText(error)
              );
    File_doneExtendedAttributes(&fileExtendedAttributeList);
    String_delete(fileName);
    String_delete(linkName);
    return error;
  }
  DEBUG_TESTCODE() { Archive_closeEntry(&sourceArchiveEntryInfo); File_doneExtendedAttributes(&fileExtendedAttributeList); String_delete(fileName); String_delete(linkName); return DEBUG_TESTCODE_ERROR(); }

  printInfo(1,"  Convert link      '%s'...",String_cString(linkName));

  // set new crypt settings
  if (Configuration_isCommandLineOptionSet(globalOptions.cryptAlgorithms)) cryptAlgorithm = newJobOptions->cryptAlgorithms[0];
  if (Configuration_isCommandLineOptionSet(globalOptions.cryptAlgorithms) || Configuration_isCommandLineOptionSet(&globalOptions.cryptNewPassword))
  {
    cryptSalt = NULL;
    cryptKey  = NULL;
  }

  // create new link entry
  ArchiveEntryInfo destinationArchiveEntryInfo;
  error = Archive_newLinkEntry(&destinationArchiveEntryInfo,
                               destinationArchiveHandle,
                               cryptAlgorithm,
                               cryptSalt,
                               cryptKey,
                               linkName,
                               fileName,
                               &fileInfo,
                               &fileExtendedAttributeList
                              );
  if (error != ERROR_NONE)
  {
    printInfo(1,"FAIL\n");
    printError(_("cannot create new archive link entry '%s' (error: %s)"),
               String_cString(linkName),
               Error_getText(error)
              );
    (void)Archive_closeEntry(&sourceArchiveEntryInfo);
    File_doneExtendedAttributes(&fileExtendedAttributeList);
    String_delete(fileName);
    String_delete(linkName);
    return error;
  }

  // close destination archive entry
  error = Archive_closeEntry(&destinationArchiveEntryInfo);
  if (error != ERROR_NONE)
  {
    printInfo(1,"FAIL\n");
    printError(_("cannot close archive link entry '%s' (error: %s)!"),
               String_cString(linkName),
               Error_getText(error)
              );
    (void)Archive_closeEntry(&sourceArchiveEntryInfo);
    File_doneExtendedAttributes(&fileExtendedAttributeList);
    String_delete(fileName);
    String_delete(linkName);
    return error;
  }

  printInfo(1,"OK\n");

  // check if all data read
  if (!Archive_eofData(&sourceArchiveEntryInfo))
  {
    printWarning(_("unexpected data at end of link entry '%s'"),String_cString(linkName));
  }

  // close source archive entry
  error = Archive_closeEntry(&sourceArchiveEntryInfo);
  if (error != ERROR_NONE)
  {
    printWarning(_("close 'link' entry fail (error: %s)"),Error_getText(error));
  }

  // free resources
  File_doneExtendedAttributes(&fileExtendedAttributeList);
  String_delete(fileName);
  String_delete(linkName);

  return ERROR_NONE;
}

/***********************************************************************\
* Name   : convertHardLinkEntry
* Purpose: convert a hard link entry in archive
* Input  : sourceArchiveHandle      - source archive handle
*          destinationArchiveHandle - destination archive handle
*          newJobOptions            - new job options
*          buffer                   - buffer for temporary data
*          bufferSize               - size of data buffer
* Output : -
* Return : ERROR_NONE or error code
* Notes  : -
\***********************************************************************/

LOCAL Errors convertHardLinkEntry(ArchiveHandle    *sourceArchiveHandle,
                                  ArchiveHandle    *destinationArchiveHandle,
                                  const JobOptions *newJobOptions,
                                  byte             *buffer,
                                  uint             bufferSize
                                 )
{
  Errors error;

  // read source hard link entry
  ArchiveFlags              archiveFlags;
  ArchiveEntryInfo          sourceArchiveEntryInfo;
  CompressAlgorithms        deltaCompressAlgorithm,byteCompressAlgorithm;
  CryptAlgorithms           cryptAlgorithm;
  const CryptSalt           *cryptSalt;
  const CryptKey            *cryptKey;
  StringList                fileNameList;
  StringList_init(&fileNameList);
  FileInfo                  fileInfo;
  FileExtendedAttributeList fileExtendedAttributeList;
  File_initExtendedAttributes(&fileExtendedAttributeList);
  uint64                    fragmentOffset,fragmentSize;
  error = Archive_readHardLinkEntry(&sourceArchiveEntryInfo,
                                    sourceArchiveHandle,
                                    &deltaCompressAlgorithm,
                                    &byteCompressAlgorithm,
                                    NULL,  // cryptType,
                                    &cryptAlgorithm,
                                    &cryptSalt,
                                    &cryptKey,
                                    &fileNameList,
                                    &fileInfo,
                                    &fileExtendedAttributeList,
                                    NULL,  // deltaSourceName
                                    NULL,  // deltaSourceSize
                                    &fragmentOffset,
                                    &fragmentSize
                                   );
  if (error != ERROR_NONE)
  {
    printError(_("cannot read 'hard link' entry from storage '%s' (error: %s)!"),
               String_cString(sourceArchiveHandle->printableStorageName),
               Error_getText(error)
              );
    File_doneExtendedAttributes(&fileExtendedAttributeList);
    StringList_done(&fileNameList);
    return error;
  }
  DEBUG_TESTCODE() { Archive_closeEntry(&sourceArchiveEntryInfo); File_doneExtendedAttributes(&fileExtendedAttributeList); StringList_done(&fileNameList); return DEBUG_TESTCODE_ERROR(); }

  // get size/fragment info
  char sizeString[32];
  ArchiveEntryInfo destinationArchiveEntryInfo;
  if (globalOptions.humanFormatFlag)
  {
    getHumanSizeString(sizeString,sizeof(sizeString),fileInfo.size);
  }
  else
  {
    stringFormat(sizeString,sizeof(sizeString),"%"PRIu64,fileInfo.size);
  }
  char fragmentString[256];
  stringClear(fragmentString);
  if (fragmentSize < fileInfo.size)
  {
    stringFormat(fragmentString,sizeof(fragmentString),", fragment %"PRIu64"..%"PRIu64,fragmentOffset,fragmentOffset+fragmentSize-1LL);
  }
  printInfo(1,"  Convert hard link '%s' (%s bytes%s)...",String_cString(StringList_first(&fileNameList,NULL)),sizeString,fragmentString);

  // set new compression, crypt settings
  if (Configuration_isCommandLineOptionSet(&globalOptions.compressAlgorithms)) byteCompressAlgorithm = newJobOptions->compressAlgorithms.byte;
  if (Configuration_isCommandLineOptionSet(globalOptions.cryptAlgorithms    )) cryptAlgorithm        = newJobOptions->cryptAlgorithms[0];
  if (Configuration_isCommandLineOptionSet(globalOptions.cryptAlgorithms) || Configuration_isCommandLineOptionSet(&globalOptions.cryptNewPassword))
  {
    cryptSalt = NULL;
    cryptKey  = NULL;
  }

  archiveFlags = ARCHIVE_FLAG_NONE;

  // check if file data should be byte compressed
  if (   (fileInfo.size > globalOptions.compressMinFileSize)
//TODO
//      && !PatternList_match(createInfo->compressExcludePatternList,fileName,PATTERN_MATCH_MODE_EXACT)
     )
  {
     archiveFlags |= ARCHIVE_FLAG_TRY_BYTE_COMPRESS;
  }

  // create new hard link entry
  error = Archive_newHardLinkEntry(&destinationArchiveEntryInfo,
                                   destinationArchiveHandle,
                                   deltaCompressAlgorithm,
                                   byteCompressAlgorithm,
                                   cryptAlgorithm,
                                   cryptSalt,
                                   cryptKey,
                                   &fileNameList,
                                   &fileInfo,
                                   &fileExtendedAttributeList,
                                   fragmentOffset,
                                   fragmentSize,
                                   archiveFlags
                                  );
  if (error != ERROR_NONE)
  {
    printError(_("cannot create new archive hardlink entry '%s' (error: %s)"),
               String_cString(StringList_first(&fileNameList,NULL)),
               Error_getText(error)
              );
    (void)Archive_closeEntry(&sourceArchiveEntryInfo);
    File_doneExtendedAttributes(&fileExtendedAttributeList);
    StringList_done(&fileNameList);
    return error;
  }
  DEBUG_TESTCODE() { Archive_closeEntry(&destinationArchiveEntryInfo); Archive_closeEntry(&sourceArchiveEntryInfo); File_doneExtendedAttributes(&fileExtendedAttributeList); StringList_done(&fileNameList); return DEBUG_TESTCODE_ERROR(); }

  // convert archive and hard link content
  uint64 length = 0LL;
  while (length < fragmentSize)
  {
    ulong bufferLength = (ulong)MIN(fragmentSize-length,bufferSize);

    // read source archive
    error = Archive_readData(&sourceArchiveEntryInfo,buffer,bufferLength);
    if (error != ERROR_NONE)
    {
      printInfo(1,"FAIL!\n");
      printError(_("cannot read content of 'hard link' entry '%s' (error: %s)!"),
                 String_cString(StringList_first(&fileNameList,NULL)),
                 Error_getText(error)
                );
      break;
    }
    DEBUG_TESTCODE() { error = DEBUG_TESTCODE_ERROR(); break; }

    // write data to destination archive
    error = Archive_writeData(&destinationArchiveEntryInfo,buffer,bufferLength,1);
    if (error != ERROR_NONE)
    {
      printInfo(0,"FAIL!\n");
      printError(_("cannot write content of 'hard link' entry '%s' (error: %s)!"),
                 String_cString(StringList_first(&fileNameList,NULL)),
                 Error_getText(error)
                );
      break;
    }
    DEBUG_TESTCODE() { error = DEBUG_TESTCODE_ERROR(); break; }

    length += (uint64)bufferLength;

    printInfo(2,"%3d%%\b\b\b\b",(uint)((length*100LL)/fragmentSize));
  }
  if (error != ERROR_NONE)
  {
    (void)Archive_closeEntry(&destinationArchiveEntryInfo);
    (void)Archive_closeEntry(&sourceArchiveEntryInfo);
    File_doneExtendedAttributes(&fileExtendedAttributeList);
    StringList_done(&fileNameList);
    return error;
  }
  printInfo(2,"    \b\b\b\b");

  // close destination archive entry
  error = Archive_closeEntry(&destinationArchiveEntryInfo);
  if (error != ERROR_NONE)
  {
    printInfo(1,"FAIL\n");
    printError(_("cannot close archive hard link entry '%s' (error: %s)!"),
               String_cString(StringList_first(&fileNameList,NULL)),
               Error_getText(error)
              );
    (void)Archive_closeEntry(&sourceArchiveEntryInfo);
    File_doneExtendedAttributes(&fileExtendedAttributeList);
    StringList_done(&fileNameList);
    return error;
  }

  printInfo(1,"OK\n");

  /* check if all data read.
     Note: it is not possible to check if all data is read when
     compression is used. The decompressor may not be at the end
     of a compressed data chunk even compressed data is _not_
     corrupt.
  */
  if (   !Compress_isCompressed(deltaCompressAlgorithm)
      && !Compress_isCompressed(byteCompressAlgorithm)
      && !Archive_eofData(&sourceArchiveEntryInfo))
  {
    printWarning(_("unexpected data at end of hard link entry '%s'"),String_cString(StringList_first(&fileNameList,NULL)));
  }

  // close source archive entry
  error = Archive_closeEntry(&sourceArchiveEntryInfo);
  if (error != ERROR_NONE)
  {
    printWarning(_("close 'hard link' entry fail (error: %s)"),Error_getText(error));
  }

  // free resources
  File_doneExtendedAttributes(&fileExtendedAttributeList);
  StringList_done(&fileNameList);

  return ERROR_NONE;
}

/***********************************************************************\
* Name   : convertSpecialEntry
* Purpose: convert a special entry in archive
* Input  : sourceArchiveHandle      - source archive handle
*          destinationArchiveHandle - destination archive handle
*          newJobOptions            - new job options
* Output : -
* Return : ERROR_NONE or error code
* Notes  : -
\***********************************************************************/

LOCAL Errors convertSpecialEntry(ArchiveHandle    *sourceArchiveHandle,
                                 ArchiveHandle    *destinationArchiveHandle,
                                 const JobOptions *newJobOptions
                                )
{
  Errors error;

  // read source special entry
  ArchiveEntryInfo          sourceArchiveEntryInfo;
  CryptAlgorithms           cryptAlgorithm;
  const CryptSalt           *cryptSalt;
  const CryptKey            *cryptKey;
  String                    fileName = String_new();
  FileInfo                  fileInfo;
  FileExtendedAttributeList fileExtendedAttributeList;
  File_initExtendedAttributes(&fileExtendedAttributeList);
  error = Archive_readSpecialEntry(&sourceArchiveEntryInfo,
                                   sourceArchiveHandle,
                                   NULL,  // cryptType,
                                   &cryptAlgorithm,
                                   &cryptSalt,
                                   &cryptKey,
                                   fileName,
                                   &fileInfo,
                                   &fileExtendedAttributeList
                                  );
  if (error != ERROR_NONE)
  {
    printError(_("cannot read 'special' entry from storage '%s' (error: %s)!"),
               String_cString(sourceArchiveHandle->printableStorageName),
               Error_getText(error)
              );
    File_doneExtendedAttributes(&fileExtendedAttributeList);
    String_delete(fileName);
    return error;
  }
  DEBUG_TESTCODE() { Archive_closeEntry(&sourceArchiveEntryInfo); File_doneExtendedAttributes(&fileExtendedAttributeList); String_delete(fileName); return DEBUG_TESTCODE_ERROR(); }

  printInfo(1,"  Convert special   '%s'...",String_cString(fileName));

  // set new crypt settings
  if (Configuration_isCommandLineOptionSet(globalOptions.cryptAlgorithms)) cryptAlgorithm = newJobOptions->cryptAlgorithms[0];
  if (Configuration_isCommandLineOptionSet(globalOptions.cryptAlgorithms) || Configuration_isCommandLineOptionSet(&globalOptions.cryptNewPassword))
  {
    cryptSalt = NULL;
    cryptKey  = NULL;
  }

  // create new special entry
  ArchiveEntryInfo destinationArchiveEntryInfo;
  error = Archive_newSpecialEntry(&destinationArchiveEntryInfo,
                                  destinationArchiveHandle,
                                  cryptAlgorithm,
                                  cryptSalt,
                                  cryptKey,
                                  fileName,
                                  &fileInfo,
                                  &fileExtendedAttributeList
                                 );
  if (error != ERROR_NONE)
  {
    printError(_("cannot create new archive special entry '%s' (error: %s)"),
               String_cString(fileName),
               Error_getText(error)
              );
    (void)Archive_closeEntry(&sourceArchiveEntryInfo);
    File_doneExtendedAttributes(&fileExtendedAttributeList);
    String_delete(fileName);
    return error;
  }
  DEBUG_TESTCODE() { Archive_closeEntry(&destinationArchiveEntryInfo); (void)Archive_closeEntry(&sourceArchiveEntryInfo); File_doneExtendedAttributes(&fileExtendedAttributeList); String_delete(fileName); return DEBUG_TESTCODE_ERROR(); }

  // close destination archive entry
  error = Archive_closeEntry(&destinationArchiveEntryInfo);
  if (error != ERROR_NONE)
  {
    printInfo(1,"FAIL\n");
    printError(_("cannot close archive special entry '%s' (error: %s)!"),
               String_cString(fileName),
               Error_getText(error)
              );
    (void)Archive_closeEntry(&sourceArchiveEntryInfo);
    File_doneExtendedAttributes(&fileExtendedAttributeList);
    String_delete(fileName);
    return error;
  }

  printInfo(1,"OK\n");

  // check if all data read
  if (!Archive_eofData(&sourceArchiveEntryInfo))
  {
    printWarning(_("unexpected data at end of special entry '%s'"),String_cString(fileName));
  }

  // close source archive entry
  error = Archive_closeEntry(&sourceArchiveEntryInfo);
  if (error != ERROR_NONE)
  {
    printWarning(_("close 'special' entry fail (error: %s)"),Error_getText(error));
  }

  // free resources
  File_doneExtendedAttributes(&fileExtendedAttributeList);
  String_delete(fileName);

  return ERROR_NONE;
}

/***********************************************************************\
* Name   : convertMetaEntry
* Purpose: convert a meta entry in archive
* Input  : sourceArchiveHandle      - source archive handle
*          destinationArchiveHandle - destination archive handle
*          newJobUUID               - new job UUID or NULL
*          newEntityUUID            - new entity UUID or NULL
*          newCreatedDateTime       - new created date/time or 0
*          newJobOptions            - new job options or NULL
* Output : -
* Return : ERROR_NONE or error code
* Notes  : -
\***********************************************************************/

LOCAL Errors convertMetaEntry(ArchiveHandle    *sourceArchiveHandle,
                              ArchiveHandle    *destinationArchiveHandle,
                              const char       *newJobUUID,
                              const char       *newEntityUUID,
                              uint64           newCreatedDateTime,
                              const JobOptions *newJobOptions
                             )
{
  Errors error;

  // read source meta entry
  ArchiveEntryInfo sourceArchiveEntryInfo;
  CryptAlgorithms  cryptAlgorithm;
  const CryptSalt  *cryptSalt;
  const CryptKey   *cryptKey;
  String           hostName = String_new();
  String           userName = String_new();
  StaticString     (jobUUID,MISC_UUID_STRING_LENGTH);
  StaticString     (entityUUID,MISC_UUID_STRING_LENGTH);
  ArchiveTypes     archiveType;
  uint64           createdDateTime;
  String           comment  = String_new();
  error = Archive_readMetaEntry(&sourceArchiveEntryInfo,
                                sourceArchiveHandle,
                                NULL,  // cryptType,
                                &cryptAlgorithm,
                                &cryptSalt,
                                &cryptKey,
                                hostName,
                                userName,
                                jobUUID,
                                entityUUID,
                                &archiveType,
                                &createdDateTime,
                                comment
                               );
  if (error != ERROR_NONE)
  {
    printError(_("cannot read 'meta' entry from storage '%s' (error: %s)!"),
               String_cString(sourceArchiveHandle->printableStorageName),
               Error_getText(error)
              );
    String_delete(comment);
    String_delete(userName);
    String_delete(hostName);
    return error;
  }
  DEBUG_TESTCODE() { Archive_closeEntry(&sourceArchiveEntryInfo); String_delete(comment); String_delete(userName); String_delete(hostName); return DEBUG_TESTCODE_ERROR(); }

  printInfo(1,"  Convert meta      ...");

  // set new job UUID, schedule UUOD, created date/time comment
  if (Configuration_isCommandLineOptionSet(globalOptions.cryptAlgorithms)) cryptAlgorithm = newJobOptions->cryptAlgorithms[0];
  if (Configuration_isCommandLineOptionSet(globalOptions.cryptAlgorithms) || Configuration_isCommandLineOptionSet(&globalOptions.cryptNewPassword))
  {
    cryptSalt = NULL;
    cryptKey  = NULL;
  }
  if (!stringIsEmpty(newJobUUID)) String_setCString(jobUUID,newJobUUID);
  if (!stringIsEmpty(newEntityUUID)) String_setCString(entityUUID,newEntityUUID);
  if (newCreatedDateTime != 0LL) createdDateTime = newCreatedDateTime;
  if (Configuration_isCommandLineOptionSet(&globalOptions.comment)) String_set(comment,newJobOptions->comment);

  // create new meta entry
  ArchiveEntryInfo destinationArchiveEntryInfo;
  error = Archive_newMetaEntry(&destinationArchiveEntryInfo,
                               destinationArchiveHandle,
                               cryptAlgorithm,
                               cryptSalt,
                               cryptKey,
                               String_cString(hostName),
                               String_cString(userName),
                               String_cString(jobUUID),
                               String_cString(entityUUID),
                               archiveType,
                               createdDateTime,
                               comment
                              );
  if (error != ERROR_NONE)
  {
    printError(_("cannot create new archive 'meta' entry (error: %s)"),
               Error_getText(error)
              );
    (void)Archive_closeEntry(&sourceArchiveEntryInfo);
    String_delete(comment);
    String_delete(userName);
    String_delete(hostName);
    return error;
  }
  DEBUG_TESTCODE() { Archive_closeEntry(&destinationArchiveEntryInfo); (void)Archive_closeEntry(&sourceArchiveEntryInfo); String_delete(comment); String_delete(userName); String_delete(hostName); return DEBUG_TESTCODE_ERROR(); }

  // close destination archive entry
  error = Archive_closeEntry(&destinationArchiveEntryInfo);
  if (error != ERROR_NONE)
  {
    printInfo(1,"FAIL\n");
    printError(_("cannot close archive 'meta' entry (error: %s)!"),
               Error_getText(error)
              );
    (void)Archive_closeEntry(&sourceArchiveEntryInfo);
    String_delete(comment);
    String_delete(userName);
    String_delete(hostName);
    return error;
  }

  printInfo(1,"OK\n");

  // check if all data read
  if (!Archive_eofData(&sourceArchiveEntryInfo))
  {
    printWarning(_("unexpected data at end of 'meta' entry"));
  }

  // close source archive entry
  error = Archive_closeEntry(&sourceArchiveEntryInfo);
  if (error != ERROR_NONE)
  {
    printWarning(_("close 'meta' entry fail (error: %s)"),Error_getText(error));
  }

  // free resources
  String_delete(comment);
  String_delete(userName);
  String_delete(hostName);

  return ERROR_NONE;
}

/***********************************************************************\
* Name   : convertEntry
* Purpose: convert single entry
* Input  : sourceArchiveHandle - source archive handle
*          archiveEntryType    - archive entry type
*          convertInfo         - convert info
*          buffer              - buffer for temporary data
*          bufferSize          - size of data buffer
* Output : -
* Return : ERROR_NONE or error code
* Notes  : -
\***********************************************************************/

LOCAL Errors convertEntry(ArchiveHandle     *sourceArchiveHandle,
                          ArchiveEntryTypes archiveEntryType,
                          ConvertInfo       *convertInfo,
                          byte              *buffer,
                          uint              bufferSize
                         )
{
  assert(sourceArchiveHandle != NULL);
  assert(convertInfo != NULL);
  assert(buffer != NULL);

  Errors error = ERROR_UNKNOWN;
  switch (archiveEntryType)
  {
    case ARCHIVE_ENTRY_TYPE_NONE:
      #ifndef NDEBUG
        HALT_INTERNAL_ERROR_UNREACHABLE();
      #endif /* NDEBUG */
      break; /* not reached */
    case ARCHIVE_ENTRY_TYPE_FILE:
      error = convertFileEntry(sourceArchiveHandle,
                               &convertInfo->destinationArchiveHandle,
                               convertInfo->newJobOptions,
                               buffer,
                               bufferSize
                              );
      break;
    case ARCHIVE_ENTRY_TYPE_IMAGE:
      error = convertImageEntry(sourceArchiveHandle,
                                &convertInfo->destinationArchiveHandle,
                                convertInfo->newJobOptions,
                                buffer,
                                bufferSize
                               );
      break;
    case ARCHIVE_ENTRY_TYPE_DIRECTORY:
      error = convertDirectoryEntry(sourceArchiveHandle,
                                    &convertInfo->destinationArchiveHandle,
                                    convertInfo->newJobOptions
                                   );
      break;
    case ARCHIVE_ENTRY_TYPE_LINK:
      error = convertLinkEntry(sourceArchiveHandle,
                               &convertInfo->destinationArchiveHandle,
                               convertInfo->newJobOptions
                              );
      break;
    case ARCHIVE_ENTRY_TYPE_HARDLINK:
      error = convertHardLinkEntry(sourceArchiveHandle,
                                   &convertInfo->destinationArchiveHandle,
                                   convertInfo->newJobOptions,
                                   buffer,
                                   bufferSize
                                  );
      break;
    case ARCHIVE_ENTRY_TYPE_SPECIAL:
      error = convertSpecialEntry(sourceArchiveHandle,
                                  &convertInfo->destinationArchiveHandle,
                                  convertInfo->newJobOptions
                                 );
      break;
    case ARCHIVE_ENTRY_TYPE_META:
      error = convertMetaEntry(sourceArchiveHandle,
                               &convertInfo->destinationArchiveHandle,
                               convertInfo->newJobUUID,
                               convertInfo->newEntityUUID,
                               convertInfo->newCreatedDateTime,
                               convertInfo->newJobOptions
                              );
      break;
    case ARCHIVE_ENTRY_TYPE_SALT:
    case ARCHIVE_ENTRY_TYPE_KEY:
      assert(!Configuration_isCommandLineOptionSet(globalOptions.cryptAlgorithms));
      error = Archive_transferEntry(sourceArchiveHandle,
                                    &convertInfo->destinationArchiveHandle
                                   );
      break;
    case ARCHIVE_ENTRY_TYPE_SIGNATURE:
      #ifndef NDEBUG
        HALT_INTERNAL_ERROR_UNREACHABLE();
      #else
        error = Archive_skipNextEntry(sourceArchiveHandle);
      #endif /* NDEBUG */
      break;
    case ARCHIVE_ENTRY_TYPE_UNKNOWN:
      #ifndef NDEBUG
        HALT_INTERNAL_ERROR_UNREACHABLE();
      #endif /* NDEBUG */
      break; /* not reached */
  }

  return error;
}

/***********************************************************************\
* Name   : convertThreadCode
* Purpose: convert worker thread
* Input  : convertInfo - convert info structure
* Output : -
* Return : -
* Notes  : -
\***********************************************************************/

LOCAL void convertThreadCode(ConvertInfo *convertInfo)
{
  assert(convertInfo != NULL);

  // convert entries
  byte *buffer = (byte*)malloc(BUFFER_SIZE);
  if (buffer == NULL)
  {
    HALT_INSUFFICIENT_MEMORY();
  }
  ArchiveHandle sourceArchiveHandle;
  uint          archiveIndex = 0;
  EntryMsg      entryMsg;
  while (MsgQueue_get(&convertInfo->entryMsgQueue,&entryMsg,NULL,sizeof(entryMsg),WAIT_FOREVER))
  {
    assert(entryMsg.archiveHandle != NULL);
    assert(entryMsg.archiveCryptInfo != NULL);
    if (   ((convertInfo->failError == ERROR_NONE) || !convertInfo->newJobOptions->noStopOnErrorFlag)
//TODO
//         && !isAborted(convertInfo)
       )
    {
      Errors error;

      if (archiveIndex < entryMsg.archiveIndex)
      {
        // open source archive
        error = Archive_openHandle(&sourceArchiveHandle,
                                   entryMsg.archiveHandle
                                  );
        if (error != ERROR_NONE)
        {
          printError(_("cannot open archive '%s' (error: %s)!"),
                     String_cString(entryMsg.archiveHandle->printableStorageName),
                     Error_getText(error)
                    );
          if (convertInfo->failError == ERROR_NONE) convertInfo->failError = error;
          freeEntryMsg(&entryMsg,NULL);
          break;
        }

        // store current archive index
        archiveIndex = entryMsg.archiveIndex;
      }

// TODO: required?
      // set crypt salt, crypt key derive type, and crypt mode
      Archive_setCryptInfo(&sourceArchiveHandle,entryMsg.archiveCryptInfo);
      if (!Configuration_isCommandLineOptionSet(globalOptions.cryptAlgorithms))
      {
        Archive_setCryptInfo(&convertInfo->destinationArchiveHandle,entryMsg.archiveCryptInfo);
      }

      // seek to start of entry
      error = Archive_seek(&sourceArchiveHandle,entryMsg.offset);
      if (error != ERROR_NONE)
      {
        printError(_("cannot read storage '%s' (error: %s)!"),
                   String_cString(sourceArchiveHandle.printableStorageName),
                   Error_getText(error)
                  );
        if (convertInfo->failError == ERROR_NONE) convertInfo->failError = error;
        freeEntryMsg(&entryMsg,NULL);
        break;
      }

      error = convertEntry(&sourceArchiveHandle,
                           entryMsg.archiveEntryType,
                           convertInfo,
                           buffer,
                           BUFFER_SIZE
                          );
      if (error != ERROR_NONE)
      {
        if (convertInfo->failError == ERROR_NONE) convertInfo->failError = error;
        freeEntryMsg(&entryMsg,NULL);
        break;
      }
    }

    freeEntryMsg(&entryMsg,NULL);
  }

  // close archive
  if (archiveIndex != 0)
  {
    Archive_close(&sourceArchiveHandle,FALSE);
  }

  free(buffer);

  // discard processing all other entries
  while (MsgQueue_get(&convertInfo->entryMsgQueue,&entryMsg,NULL,sizeof(entryMsg),WAIT_FOREVER))
  {
    assert(entryMsg.archiveHandle != NULL);
    assert(entryMsg.archiveCryptInfo != NULL);

    freeEntryMsg(&entryMsg,NULL);
  }

  // free resources
}

/***********************************************************************\
* Name   : convertArchive
* Purpose: convert archive
* Input  : convertInfo      - convert info
*          storageSpecifier - storage specifier
*          archiveName      - archive name (can be NULL)
* Output : -
* Return : -
* Notes  : -
\***********************************************************************/

LOCAL Errors convertArchive(ConvertInfo      *convertInfo,
                            StorageSpecifier *storageSpecifier,
                            ConstString      archiveName
                           )
{
  Errors error;

  assert(convertInfo != NULL);
  assert(convertInfo->newJobOptions != NULL);
  assert(storageSpecifier != NULL);

  // init variables
  AutoFreeList autoFreeList;
  AutoFree_init(&autoFreeList);

  // Note: still not supported
  if (convertInfo->newJobOptions->archivePartSize > 0LL)
  {
    printWarning(_("Archive part size not supported for convert - ignored"));
  }
//TODO
//  convertInfo->newJobOptions->archivePartSize = 0LL;

  // set printable storage name
  String printableStorageName = Storage_getPrintableName(String_new(),storageSpecifier,archiveName);
  AUTOFREE_ADD(&autoFreeList,printableStorageName,{ String_delete(printableStorageName); });

  // init source storage
  error = Storage_init(&convertInfo->sourceStorageInfo,
                       NULL,  // masterIO
                       storageSpecifier,
                       convertInfo->newJobOptions,
                       &globalOptions.maxBandWidthList,
                       SERVER_CONNECTION_PRIORITY_HIGH,
                       CALLBACK_(NULL,NULL),  // storageUpdateProgress
                       CALLBACK_(NULL,NULL),  // getPassword
                       CALLBACK_(NULL,NULL),  // requestVolume
                       CALLBACK_(NULL,NULL),  // isPause
                       CALLBACK_(NULL,NULL),  // isAborted
                       NULL  // logHandle
                      );
  if (error != ERROR_NONE)
  {
    printError(_("cannot initialize storage '%s' (error: %s)!"),
               String_cString(printableStorageName),
               Error_getText(error)
              );
    AutoFree_cleanup(&autoFreeList);
    return error;
  }
  DEBUG_TESTCODE() { AutoFree_cleanup(&autoFreeList); return DEBUG_TESTCODE_ERROR(); }
  AUTOFREE_ADD(&autoFreeList,&convertInfo->sourceStorageInfo,{ (void)Storage_done(&convertInfo->sourceStorageInfo); });

  // check if storage exists
  if (!Storage_exists(&convertInfo->sourceStorageInfo,archiveName))
  {
    printError(_("storage not found '%s'!"),
               String_cString(printableStorageName)
              );
    AutoFree_cleanup(&autoFreeList);
    return ERROR_ARCHIVE_NOT_FOUND;
  }

  // open source archive
  ArchiveHandle sourceArchiveHandle;
  error = Archive_open(&sourceArchiveHandle,
                       &convertInfo->sourceStorageInfo,
                       archiveName,
                       NULL,  // deltaSourceList,
                       ARCHIVE_FLAG_NONE,
                       CALLBACK_(convertInfo->getNamePasswordFunction,convertInfo->getNamePasswordUserData),
                       convertInfo->logHandle
                      );
  if (error != ERROR_NONE)
  {
    printError(_("cannot open archive '%s' (error: %s)!"),
               String_cString(printableStorageName),
               Error_getText(error)
              );
    AutoFree_cleanup(&autoFreeList);
    return error;
  }
  DEBUG_TESTCODE() { Archive_close(&sourceArchiveHandle,FALSE); AutoFree_cleanup(&autoFreeList); return DEBUG_TESTCODE_ERROR(); }
  AUTOFREE_ADD(&autoFreeList,&sourceArchiveHandle,{ Archive_close(&sourceArchiveHandle,FALSE); });

  // check signatures
  CryptSignatureStates sourceAllCryptSignatureState = CRYPT_SIGNATURE_STATE_NONE;
  uint64               sourceLastSignatureOffset    = Archive_tell(&sourceArchiveHandle);
  if (!convertInfo->newJobOptions->skipVerifySignaturesFlag)
  {
    error = Archive_verifySignatures(&sourceArchiveHandle,
                                     &sourceAllCryptSignatureState
                                    );
    if (error != ERROR_NONE)
    {
      if (!convertInfo->newJobOptions->forceVerifySignaturesFlag && (Error_getCode(error) == ERROR_CODE_NO_PUBLIC_SIGNATURE_KEY))
      {
        sourceAllCryptSignatureState = CRYPT_SIGNATURE_STATE_SKIPPED;
      }
      else
      {
        // signature error
        AutoFree_cleanup(&autoFreeList);
        return error;
      }
    }
    if (!Crypt_isValidSignatureState(sourceAllCryptSignatureState))
    {
      if (convertInfo->newJobOptions->forceVerifySignaturesFlag)
      {
        // signature error
        printError(_("invalid signature in '%s'!"),
                   String_cString(printableStorageName)
                  );
        AutoFree_cleanup(&autoFreeList);
        return ERROR_INVALID_SIGNATURE;
      }
      else
      {
        // print signature warning
        printWarning(_("invalid signature in '%s'!"),
                     String_cString(printableStorageName)
                    );
      }
    }
  }

  // init destination storage
  error = Storage_init(&convertInfo->destinationStorageInfo,
                       NULL,  // masterIO
                       storageSpecifier,
                       convertInfo->newJobOptions,
                       &globalOptions.maxBandWidthList,
                       SERVER_CONNECTION_PRIORITY_HIGH,
                       CALLBACK_(NULL,NULL),  // storageUpdateProgress
                       CALLBACK_(NULL,NULL),  // getPassword
                       CALLBACK_(NULL,NULL),  // requestVolume
                       CALLBACK_(NULL,NULL),  // isPause
                       CALLBACK_(NULL,NULL),  // isAborted
                       NULL  // logHandle
                      );
  if (error != ERROR_NONE)
  {
    printError(_("cannot initialize storage '%s' (error: %s)!"),
               String_cString(printableStorageName),
               Error_getText(error)
              );
    AutoFree_cleanup(&autoFreeList);
    return error;
  }
  DEBUG_TESTCODE() { AutoFree_cleanup(&autoFreeList); return DEBUG_TESTCODE_ERROR(); }
  AUTOFREE_ADD(&autoFreeList,&convertInfo->destinationStorageInfo,{ (void)Storage_done(&convertInfo->destinationStorageInfo); });

  // create destination archive name
  String baseName = File_getBaseName(String_new(),(archiveName != NULL) ? archiveName : storageSpecifier->archiveName,TRUE);
  if (!String_isEmpty(convertInfo->newJobOptions->destination))
  {
    File_setFileName(convertInfo->archiveName,convertInfo->newJobOptions->destination);
    File_appendFileName(convertInfo->archiveName,baseName);
  }
  else
  {
    File_setFileName(convertInfo->archiveName,(archiveName != NULL) ? archiveName : storageSpecifier->archiveName);
  }
  String_delete(baseName);

  // create new archive
  ArchiveFlags archiveFlags = ARCHIVE_FLAG_SKIP_UNKNOWN_CHUNKS;
  if (Configuration_isCommandLineOptionSet(globalOptions.cryptAlgorithms))
  {
    archiveFlags |= ARCHIVE_FLAG_CREATE_SALT|ARCHIVE_FLAG_CREATE_KEY;
  }
  else
  {
    archiveFlags |= ARCHIVE_FLAG_RETURN_SALT_CHUNKS|ARCHIVE_FLAG_RETURN_KEY_CHUNKS;
  }
  if (isPrintInfo(3)) archiveFlags |= ARCHIVE_FLAG_PRINT_UNKNOWN_CHUNKS;
  error = Archive_create(&convertInfo->destinationArchiveHandle,
                         NULL,  // hostName
                         NULL,  // userName
                         &convertInfo->destinationStorageInfo,
                         NULL,  // archiveName,
                         INDEX_ID_NONE,  // uuidId,
                         INDEX_ID_NONE,  // entityId,
// TODO:
                         NULL,  // jobUUID
                         NULL,  // entityUUID
// TODO:
                         NULL,  // deltaSourceList,
                         sourceArchiveHandle.archiveType,
                         FALSE, // dryRun
                         Misc_getCurrentDateTime(),
                         Configuration_isCommandLineOptionSet(&globalOptions.cryptNewPassword)
                           ? &globalOptions.cryptNewPassword
                           : &globalOptions.cryptPassword,
                         archiveFlags,
                         CALLBACK_(NULL,NULL),  // archiveInitFunction
                         CALLBACK_(NULL,NULL),  // archiveDoneFunction
                         CALLBACK_(NULL,NULL),  // archiveGetSizeFunction
                         CALLBACK_(NULL,NULL),  // archiveTestFunction
                         CALLBACK_(archiveStore,convertInfo),
                         CALLBACK_(convertInfo->getNamePasswordFunction,convertInfo->getNamePasswordUserData),
                         convertInfo->logHandle
                        );
  if (error != ERROR_NONE)
  {
    printError(_("cannot create temporary storage for '%s' (error: %s)"),
               String_cString(printableStorageName),
               Error_getText(error)
              );
    AutoFree_cleanup(&autoFreeList);
    return error;
  }
  DEBUG_TESTCODE() { Archive_close(&convertInfo->destinationArchiveHandle,FALSE); AutoFree_cleanup(&autoFreeList); return DEBUG_TESTCODE_ERROR(); }
  AUTOFREE_ADD(&autoFreeList,&convertInfo->destinationArchiveHandle,{ Archive_close(&convertInfo->destinationArchiveHandle,FALSE); });

//TODO: really required? If convert is done for each archive storage can be done as the final step without a separated thread
  // start storage thread
  MsgQueue_reset(&convertInfo->storageMsgQueue);
  Thread storageThread;
  if (!Thread_init(&storageThread,"BAR storage",globalOptions.niceLevel,storageThreadCode,convertInfo))
  {
    HALT_FATAL_ERROR("Cannot initialize storage thread!");
  }
  AUTOFREE_ADD(&autoFreeList,&storageThread,{ MsgQueue_setEndOfMsg(&convertInfo->storageMsgQueue); Thread_join(&storageThread); Thread_done(&storageThread); });

  // start convert threads/allo0cate buffer
// TODO: cannot be done multi-threaded because chunks-ordering is important and must not be changed! do multiple storages in parallel
//  convertThreadCount = (globalOptions.maxThreads != 0) ? globalOptions.maxThreads : Thread_getNumberOfCores();
  uint convertThreadCount = 1;
  byte *buffer;
  if (convertThreadCount > 1)
  {
    MsgQueue_reset(&convertInfo->entryMsgQueue);
    for (uint i = 0; i < convertThreadCount; i++)
    {
      ThreadPool_run(&workerThreadPool,convertThreadCode,convertInfo);
    }
    buffer = NULL;
  }
  else
  {
    buffer = (byte*)malloc(BUFFER_SIZE);
    if (buffer == NULL)
    {
      HALT_INSUFFICIENT_MEMORY();
    }
    AUTOFREE_ADD(&autoFreeList,buffer,{ free(buffer); });
  }

  // output info
  printInfo(0,
            "Convert storage '%s'%s",
            String_cString(printableStorageName),
            !isPrintInfo(1) ? "..." : ":\n"
           );

  // read archive entries
  error = Archive_seek(&sourceArchiveHandle,0LL);
  if (error != ERROR_NONE)
  {
    if (convertInfo->failError == ERROR_NONE) convertInfo->failError = error;
  }
  while (   (convertInfo->failError == ERROR_NONE)
         && !Archive_eof(&sourceArchiveHandle)
        )
  {
    // get next archive entry type
    ArchiveEntryTypes archiveEntryType;
    ArchiveCryptInfo  *archiveCryptInfo;
    uint64            offset;
    uint64            size;
    error = Archive_getNextArchiveEntry(&sourceArchiveHandle,
                                        &archiveEntryType,
                                        &archiveCryptInfo,
                                        &offset,
                                        &size
                                       );
    if (error != ERROR_NONE)
    {
      printError(_("cannot read next entry from storage '%s' (error: %s)!"),
                 String_cString(printableStorageName),
                 Error_getText(error)
                );
      if (convertInfo->failError == ERROR_NONE) convertInfo->failError = error;
      break;
    }
    DEBUG_TESTCODE() { convertInfo->failError = DEBUG_TESTCODE_ERROR(); break; }

    // convert entries
//fprintf(stderr,"%s:%d: %s archiveEntryType=%s %llu %llu\n",__FILE__,__LINE__,String_cString(printableStorageName),Archive_archiveEntryTypeToString(archiveEntryType),offset,size);
    if (archiveEntryType != ARCHIVE_ENTRY_TYPE_SIGNATURE)
    {
      if (convertThreadCount > 1)
      {
        // send entry to convert threads
//TODO: increment on multiple archives and when threads are not restarted each time
        EntryMsg entryMsg;
        entryMsg.archiveIndex     = 1;
        entryMsg.archiveHandle    = &sourceArchiveHandle;
        entryMsg.archiveEntryType = archiveEntryType;
        entryMsg.archiveCryptInfo = archiveCryptInfo;
        entryMsg.offset           = offset;
        if (!MsgQueue_put(&convertInfo->entryMsgQueue,&entryMsg,sizeof(entryMsg)))
        {
          HALT_INTERNAL_ERROR("Send message to convert threads fail!");
        }

        // skip entry
        error = Archive_skipNextEntry(&sourceArchiveHandle);
        if (error != ERROR_NONE)
        {
          if (convertInfo->failError == ERROR_NONE) convertInfo->failError = error;
          break;
        }
      }
      else
      {
        error = convertEntry(&sourceArchiveHandle,
                             archiveEntryType,
                             convertInfo,
                             buffer,
                             BUFFER_SIZE
                            );
      }
    }
    else
    {
      if (!convertInfo->newJobOptions->skipVerifySignaturesFlag)
      {
        // check signature
        error = Archive_verifySignatureEntry(&sourceArchiveHandle,sourceLastSignatureOffset,&sourceAllCryptSignatureState);
      }
      else
      {
        // skip signature
        error = Archive_skipNextEntry(&sourceArchiveHandle);
      }
      if (error != ERROR_NONE)
      {
        if (convertInfo->failError == ERROR_NONE) convertInfo->failError = error;
        break;
      }
      sourceLastSignatureOffset = Archive_tell(&sourceArchiveHandle);
    }
  }

  // wait for convert threads
  if (convertThreadCount > 1)
  {
    MsgQueue_setEndOfMsg(&convertInfo->entryMsgQueue);
    ThreadPool_joinAll(&workerThreadPool);
  }
  else
  {
    AUTOFREE_REMOVE(&autoFreeList,buffer);
    free(buffer);
  }

  // close destination archive
  AUTOFREE_REMOVE(&autoFreeList,&convertInfo->destinationArchiveHandle);
  error = Archive_close(&convertInfo->destinationArchiveHandle,convertInfo->failError == ERROR_NONE);
  if (error != ERROR_NONE)
  {
    printInfo(0,"FAIL\n");
    printError(_("cannot close archive '%s' (error: %s)"),
               String_cString(printableStorageName),
               Error_getText(error)
              );
    AutoFree_cleanup(&autoFreeList);
    if (convertInfo->failError == ERROR_NONE) convertInfo->failError = error;
    return error;
  }
  DEBUG_TESTCODE() { AutoFree_cleanup(&autoFreeList); return DEBUG_TESTCODE_ERROR(); }

  // wait for storage thread/free buffer
  AUTOFREE_REMOVE(&autoFreeList,&storageThread);
  MsgQueue_setEndOfMsg(&convertInfo->storageMsgQueue);
  if (!Thread_join(&storageThread))
  {
    HALT_INTERNAL_ERROR("Cannot stop storage thread!");
  }
  Thread_done(&storageThread);

  // close source archive
  AUTOFREE_REMOVE(&autoFreeList,&sourceArchiveHandle);
  (void)Archive_close(&sourceArchiveHandle,FALSE);

  // done storages
  (void)Storage_done(&convertInfo->destinationStorageInfo);
  (void)Storage_done(&convertInfo->sourceStorageInfo);

  // output info
  if (!isPrintInfo(1))
  {
    printInfo(0,
              "%s",
              (convertInfo->failError == ERROR_NONE)
                ? "OK\n"
                : "FAIL!\n"
             );
  }

  // output signature error/warning
  if (!Crypt_isValidSignatureState(sourceAllCryptSignatureState))
  {
    if (convertInfo->newJobOptions->forceVerifySignaturesFlag)
    {
      printError(_("invalid signature in '%s'!"),
                 String_cString(printableStorageName)
                );
      if (convertInfo->failError == ERROR_NONE) convertInfo->failError = ERROR_INVALID_SIGNATURE;
    }
    else
    {
      printWarning(_("invalid signature in '%s'!"),
                   String_cString(printableStorageName)
                  );
    }
  }

  // free resources
  String_delete(printableStorageName);
  AutoFree_done(&autoFreeList);

  return convertInfo->failError;
}

/*---------------------------------------------------------------------*/

Errors Command_convert(const StringList        *storageNameList,
                       const char              *newJobUUID,
                       const char              *newEntityUUID,
                       uint64                  newCreatedDateTime,
                       JobOptions              *newJobOptions,
                       GetNamePasswordFunction getNamePasswordFunction,
                       void                    *getNamePasswordUserData,
                       LogHandle               *logHandle
                      )
{
  assert(storageNameList != NULL);
  assert(newJobOptions != NULL);

  // init convert info
  ConvertInfo                convertInfo;
  initConvertInfo(&convertInfo,
                  newJobUUID,
                  newEntityUUID,
                  newCreatedDateTime,
                  newJobOptions,
                  CALLBACK_(getNamePasswordFunction,getNamePasswordUserData),
                  logHandle
                 );

  bool             someStorageFound = FALSE;
  StorageSpecifier storageSpecifier;
  Storage_initSpecifier(&storageSpecifier);
  ConstString      storageName;
  STRINGLIST_ITERATE(storageNameList,storageName)
  {
    Errors error;

    // parse storage name
    error = Storage_parseName(&storageSpecifier,storageName);
    if (error != ERROR_NONE)
    {
      printError(_("invalid storage '%s' (error: %s)!"),
                 String_cString(storageName),
                 Error_getText(error)
                );
      if (convertInfo.failError == ERROR_NONE) convertInfo.failError = error;
      continue;
    }
    DEBUG_TESTCODE() { convertInfo.failError = DEBUG_TESTCODE_ERROR(); break; }

    error = ERROR_UNKNOWN;

    // try convert archive content
    if (error != ERROR_NONE)
    {
      if (String_isEmpty(storageSpecifier.archivePatternString))
      {
        // convert archive content
        error = convertArchive(&convertInfo,
                               &storageSpecifier,
                               NULL
                              );
        if (error == ERROR_NONE)
        {
          someStorageFound = TRUE;
        }
        else
        {
          if (convertInfo.failError == ERROR_NONE) convertInfo.failError = error;
        }
      }
    }

    // try convert directory content
    if (error != ERROR_NONE)
    {
      StorageDirectoryListHandle storageDirectoryListHandle;
      error = Storage_openDirectoryList(&storageDirectoryListHandle,
                                        &storageSpecifier,
                                        NULL,  // pathName
                                        newJobOptions,
                                        SERVER_CONNECTION_PRIORITY_HIGH
                                       );
      if (error == ERROR_NONE)
      {
        String fileName = String_new();
        while (!Storage_endOfDirectoryList(&storageDirectoryListHandle))
        {
          // read next directory entry
          FileInfo fileInfo;
          error = Storage_readDirectoryList(&storageDirectoryListHandle,fileName,&fileInfo);
          if (error != ERROR_NONE)
          {
            continue;
          }

          // match pattern
          if (!String_isEmpty(storageSpecifier.archivePatternString))
          {
            if (!Pattern_match(&storageSpecifier.archivePattern,fileName,STRING_BEGIN,PATTERN_MATCH_MODE_EXACT,NULL,NULL))
            {
              continue;
            }
          }

          // convert archive content
          if (   (fileInfo.type == FILE_TYPE_FILE)
              || (fileInfo.type == FILE_TYPE_LINK)
              || (fileInfo.type == FILE_TYPE_HARDLINK)
             )
          {
            error = convertArchive(&convertInfo,
                                   &storageSpecifier,
                                   fileName
                                  );
            if (error != ERROR_NONE)
            {
              if (convertInfo.failError == ERROR_NONE) convertInfo.failError = error;
            }
          }
          someStorageFound = TRUE;
        }
        String_delete(fileName);

        Storage_closeDirectoryList(&storageDirectoryListHandle);
      }
    }

    if (error != ERROR_NONE)
    {
      if (convertInfo.failError == ERROR_NONE) convertInfo.failError = error;
      continue;
    }

    if (convertInfo.failError != ERROR_NONE) break;
  }
  if ((convertInfo.failError == ERROR_NONE) && !StringList_isEmpty(storageNameList) && !someStorageFound)
  {
    printError(_("no matching storage files found!"));
    convertInfo.failError = ERROR_FILE_NOT_FOUND_;
  }
  Storage_doneSpecifier(&storageSpecifier);

  // done convert info
  doneConvertInfo(&convertInfo);

  // output info
  if (convertInfo.failError != ERROR_NONE)
  {
    printInfo(1,tr("Convert fail: %s\n"),Error_getText(convertInfo.failError));
  }

  return convertInfo.failError;
}

#ifdef __cplusplus
  }
#endif

/* end of file */
